# restream

[![npm version](https://badge.fury.io/js/restream.svg)](https://badge.fury.io/js/restream)

Regular Expression Transform Stream for Node.js

## restream(regex: RegExp) => Transform

Create a transform stream which will buffer incoming data and push regex results
when matches can be made, i.e. when `regex.exec` returns non-null value. You will probably want to
set the `g` flag on regexes most of the time.

```js
const restream = require('restream')
const Readable = require('stream').Readable
const Writable = require('stream').Writable

// your input readable stream which outputs strings
const input = 'test-string-{12345}-{67890}'
const rs = new Readable({
    read: () => {
        rs.push(input)
        rs.push(null)
    },
})

// your output writable sream which saves incoming data
const ws = new Writable({ objectMode: true })
const data = []
ws._write = (chunk, _, next) => {
    data.push(chunk)
    next()
}

const regex = /{(\d+)}/g
const rts = restream(regex)
rs.pipe(rts).pipe(ws)

ws.once('finish', () => {
    console.log(data)
})
```

```js
[ [ '{12345}',
    '12345',
    index: 12,
    input: 'test-string-{12345}-{67890}' ],
  [ '{67890}',
    '67890',
    index: 20,
    input: 'test-string-{12345}-{67890}' ] ]
```

## restream.replaceStream(regexes: object[]) => Readable

Create a `Readable` stream which will make data available
when an incoming chunk has been updated according to the `regex` input, which can
be either a single regex object (`{ re: /regex/, replacement: 'hello-world' }`),
or an array of such objects. A `replacement` can be either a string, or a function,
which will be called by `str.replace`

```js
const restream = require('restream')
const Catchment = require('catchment')
const Readable = require('stream').Readable

function createReadable() {
    const rs = new Readable({
        read: () => {
            rs.push('Hello {{ name }}, your username is {{ user }}'
                + ' and you have {{ stars }} stars')
            rs.push(null)
        },
    })
    return rs
}

const regexes = [{
    re: /{{ user }}/,
    replacement: 'fred',
}, {
    re: /{{ name }}/g,
    replacement: 'Fred',
}, {
    re: /{{ stars }}/,
    replacement: '5',
}]

const stream = restream.replaceStream(regexes)
const rs = createReadable()
rs.pipe(stream)
const catchment = new Catchment()
stream.pipe(catchment)
catchment.promise
    .then((res) => {
        console.log(res)
    })
```

Output:

```fs
Hello Fred, your username is fred and you have 5 stars
```

### Using a function

You can replace matches using a function. See ([MDN](https://developer.mozilla.org/en/docs/Web/JavaScript/Reference/Global_Objects/String/replace#Specifying_a_function_as_a_parameter))
for more documentation.

```js
const restream = require('restream')
const Catchment = require('catchment')
const Readable = require('stream').Readable

function createReadable() {
    const rs = new Readable({
        read: () => {
            rs.push('Hello __Fred__, your username is __fred__'
                + ' and you have __5__ stars')
            rs.push(null)
        },
    })
    return rs
}

const regexes = [{
    re: /__(\S+)__/g,
    replacement: (match, p1) => {
        return `<em>${p1}</em>`
    },
}]

const stream = restream.replaceStream(regexes)
const rs = createReadable()
rs.pipe(stream)
const catchment = new Catchment()
stream.pipe(catchment)
catchment.promise
    .then((res) => {
        console.log(res)
    })
```

Output:

```fs
Hello <em>Fred</em>, your username is <em>fred</em> and you have <em>5</em> stars
```
---

2017, [Sobesednik Media](https://sobesednik.media)
